// ==UserScript==
// @name         5ch 自動スクロール（Shift+Zで切替）
// @namespace    https://example.com/
// @version      1.3.2
// @description  新着レスに自動スクロールする機能。速度調整＆ON/OFF切替可能。下方向のみスクロール。Alt+ZでON/OFF切替。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    const SCROLL_DURATION = 6000;
    const SCROLL_OFFSET = 500;

    if (window.top !== window.self) return;

    let enabled = true;
    let animationFrameId = null;
    let isUserScrolling = false;
    let toggleButton = null;

    function smoothScrollToElement(element, duration = 1000, offset = 0) {
        return new Promise((resolve) => {
            if (animationFrameId) {
                cancelAnimationFrame(animationFrameId);
                animationFrameId = null;
            }
            isUserScrolling = false;

            const elementY = element.getBoundingClientRect().top + window.scrollY;
            let targetY = elementY - offset;
            const maxScrollY = document.documentElement.scrollHeight - window.innerHeight;
            if (targetY > maxScrollY) targetY = maxScrollY;
            if (targetY < 0) targetY = 0;
            const startY = window.scrollY;

            if (targetY <= startY) {
                resolve();
                return;
            }

            const startTime = performance.now();
            function step(currentTime) {
                if (isUserScrolling) {
                    resolve();
                    animationFrameId = null;
                    return;
                }

                const elapsed = currentTime - startTime;
                const progress = Math.min(elapsed / duration, 1);
                const ease = progress < 0.5
                    ? 2 * progress * progress
                    : -1 + (4 - 2 * progress) * progress;

                window.scrollTo(0, startY + (targetY - startY) * ease);

                if (progress < 1) {
                    animationFrameId = requestAnimationFrame(step);
                } else {
                    animationFrameId = null;
                    resolve();
                }
            }

            animationFrameId = requestAnimationFrame(step);
        });
    }

    function onUserScroll() {
        if (animationFrameId !== null) {
            isUserScrolling = true;
        }
    }

    window.addEventListener('wheel', onUserScroll, { passive: true });
    window.addEventListener('touchstart', onUserScroll, { passive: true });
    window.addEventListener('keydown', (e) => {
        const keys = ['ArrowUp', 'ArrowDown', 'PageUp', 'PageDown', 'Home', 'End', ' '];
        if (keys.includes(e.key)) {
            onUserScroll();
        }

        // Shift + Z で自動スクロール ON/OFF トグル
        if (e.shiftKey && e.key.toUpperCase() === 'Z') {
            e.preventDefault();
            toggleAutoScroll();
        }
    }, { passive: false });

    function toggleAutoScroll() {
        enabled = !enabled;
        if (toggleButton) {
            toggleButton.textContent = enabled ? '自動スクロール: ON' : '自動スクロール: OFF';
            toggleButton.style.background = enabled ? '#0e0e0e' : '#2a2a2a';
        }
    }

    function createToggleButton() {
        const btn = document.createElement('button');
        toggleButton = btn;
        btn.textContent = '自動スクロール: ON';
        Object.assign(btn.style, {
            position: 'fixed',
            bottom: '6px',
            right: '70px',
            zIndex: 9999,
            padding: '8px 12px',
            background: '#0e0e0e',
            opacity: '0.5',
            color: '#6f6f6f',
            border: 'none',
            borderRadius: '4px',
            cursor: 'pointer',
            fontSize: '12px',
            boxShadow: '0 2px 6px rgba(0,0,0,0.3)'
        });
        document.body.appendChild(btn);

        btn.addEventListener('click', toggleAutoScroll);
    }

    const observer = new MutationObserver((mutations) => {
        if (!enabled) return;
        for (const mutation of mutations) {
            for (const node of mutation.addedNodes) {
                if (
                    node.nodeType === Node.ELEMENT_NODE &&
                    node.classList.contains('post') &&
                    node.getAttribute('data-date') === 'NG'
                ) {
                    smoothScrollToElement(node, SCROLL_DURATION, SCROLL_OFFSET);
                    return;
                }
            }
        }
    });

    createToggleButton();
    observer.observe(document.body, {
        childList: true,
        subtree: true,
    });

})();
